<?php

namespace App\Livewire\Auth;
use App\Helpers\Helper;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Validator;

class LoginComponent extends Component
{

    public $email, $password, $remember;

    protected $rules = [
        'email' => 'required|email',
        'password' => 'required|string|min:8',
    ];

    // Throttling check function
    protected function checkThrottle()
    {
        $key = 'login.' . $this->email;

        if (RateLimiter::tooManyAttempts($key, 5)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'email' => ["Too many login attempts. Please try again in {$seconds} seconds."],
            ]);
        }
    }

    // Login method
    public function login()
    {
        $this->validate();
        // Check if the login attempts are being throttled
        $this->checkThrottle();

        // Attempt to login
        if (Auth::attempt(['email' => $this->email, 'password' => $this->password])) {
            Helper::userLoginHistory('App\Models\User',auth()->user()->id,request()->headers->all(),request()->session()->getId());
            return $this->redirect('/dashboard',navigate:true); // Redirect after successful login
        }
        // Throttle login attempts if login fails
        RateLimiter::hit('login.' . $this->email);

        // Throw validation error on failed login
        throw ValidationException::withMessages([
            'email' => ['These credentials do not match our records.'],
        ]);
    }

    // Render method for the Livewire view
    public function render()
    {
        return view('admin.auth.login')->layout('components.layouts.guest');
    }
}
