<?php

namespace App\Livewire\Admin;
use App\Models\User;
use App\Models\Role;
use App\Models\Module;
use App\Models\Permission;
use App\Helpers\Helper;

use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Gate;
use Carbon;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;
use App\Livewire\Traits\AuthorizesRequests;

class UserComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;
    public $role;

    public $user, $name, $email, $phone, $password, $password_confirmation, $gender, $date_of_birth, $joining_date, $image,$change_password;
    public $users = [];
    public $roles = [];
    public $modules = [];
    public $user_roles = [];
    public $user_permissions = [];


    public $editModal = false;
    public $createModal = false;

    protected $listeners = ['createModal','editModal','updateChoices'];
    public function createModal(){
        $this->createModal = false;
    }
    public function editModal(){
        $this->editModal = false;
    }
    public function updateChoices($property, $value)
    {
        $this->$property = $value;
    }

    public function render()
    {
        if (!Gate::allows('view_user') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->roles = Role::all();
        $query = User::with('roles','permissions');
        if($this->search){
            $query->where('name','like', '%'.$this->search.'%')
            ->orWhere('email','like', '%'.$this->search.'%')
            ->orWhere('phone','like', '%'.$this->search.'%');
        }
        if ($this->role) {
            $query->whereHas('roles', function ($q) {
                $q->where('id', $this->role);
            });
        }
        $collection = $query->paginate($this->perPage);
        return view('admin.users.index',compact('collection'));
    }

    public function create()
    {
        if (!Gate::allows('create_user') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->roles = Role::all();
        $this->modules = Module::with('permissions')->get(); 
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal']);
    }
    
    public function store()
    {
        if (!Gate::allows('create_role') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'name'              =>  'required|string|max:255',
            'email'             =>  'required|string|email|max:50|unique:users,email,NULL,id,deleted_at,NULL',
            'phone'             =>  'required|numeric|max_digits:15|unique:users,phone,NULL,id,deleted_at,NULL',
            'gender'            =>  'required|in:Male,Female',
            'date_of_birth'     =>  'required|date_format:Y-m-d|before_or_equal:'.date('Y-m-d'),
            'joining_date'      =>  'required|date_format:Y-m-d',
            'image'             =>  'nullable|mimes:jpeg,png,jpg,webp|max:5120|exclude',
            'user_roles'        =>  'required|array|exclude',
            'user_roles.*'      =>  'exists:roles,id',
            'user_permissions'  =>  'nullable|array|exclude',
            'user_permissions.*'=>  'exists:permissions,id',
            'password'          =>  'required|string|min:8|confirmed',
        ]);
        $data['password'] = Hash::make($this->password);
        if($this->image && $this->image->isValid()){
            $data['image'] = Helper::saveFile('users',$this->image,$w=200,$h=200);
            $this->image->delete(); // Clean up the temporary file
        }else{
            $data['image'] = Helper::saveDefaultImage('users',$this->gender);
        }
        $user = User::create($data);

        if (isset($this->user_roles)) {
            $user->roles()->sync($this->user_roles);
        }
        if (isset($this->user_permissions)) {
            $user->permissions()->sync($this->user_permissions);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Role has been created successfully!']);
        $this->dispatch('hideModal',['modal'=>'createModal']);
    }
    
    public function edit($id)
    {
        if (!Gate::allows('update_user') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->editModal = true;
        $this->resetErrorBag();
        $this->user = User::with('roles','permissions')->find($id);
        $this->name = $this->user->name;
        $this->email= $this->user->email;
        $this->phone = $this->user->phone;
        $this->gender = $this->user->gender;
        $this->date_of_birth = !empty($this->user->date_of_birth) ? \Carbon\Carbon::parse($this->user->date_of_birth)->format('Y-m-d') : null;
        $this->joining_date = !empty($this->user->joining_date) ? \Carbon\Carbon::parse($this->user->joining_date)->format('Y-m-d') : null;
        $this->roles = Role::all();
        $this->modules = Module::all();
        $this->user_roles = $this->user->roles->pluck('id')->toArray();
        $this->user_permissions = $this->user->permissions->pluck('id')->toArray();
        $this->dispatch('showModal',['modal'=>'editModal']);
    }
    public function update($id)
    {
        if (!Gate::allows('update_user') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'name'              =>  'required|string|max:255',
            'email'             =>  'required|string|email|max:50|unique:users,email,'.$this->user->id.',id,deleted_at,NULL',
            'phone'             =>  'required|numeric|max_digits:15|unique:users,phone,'.$this->user->id.',id,deleted_at,NULL',
            'gender'            =>  'in:Male,Female',
            'date_of_birth'     =>  'required|date_format:Y-m-d|before_or_equal:'.date('Y-m-d'),
            'joining_date'      =>  'required|date_format:Y-m-d',
            'image'             =>  'nullable|mimes:jpeg,png,jpg,webp|max:5120|exclude',
            'user_roles'        =>  'required|array|exclude',
            'user_roles.*'      =>  'exists:roles,id',
            'user_permissions'  =>  'nullable|array|exclude',
            'user_permissions.*'=>  'exists:permissions,id',
            'change_password'   =>  'nullable|boolean|exclude',
            'password'          =>  'required_if:change_password,==,true|nullable|string|min:8|confirmed|exclude',
        ]);
       
        if (!empty($this->password)) {
            $data['password'] = Hash::make($this->password); // Hash new password if provided
        }
        if($this->image && $this->image->isValid()){
            Helper::deleteFile($this->user->image);
            $data['image'] = Helper::saveFile('users',$this->image,$w=200,$h=200);
            $this->image->delete(); // Clean up the temporary file
            $this->reset('image');
        }
        $this->user->update($data);

        if (isset($this->user_roles)) {
            $this->user->roles()->sync($this->user_roles);
        }
        if (isset($this->user_permissions)) {
            $this->user->permissions()->sync($this->user_permissions);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'User has been updated successfully!']);
        $this->dispatch('hideModal',['modal'=>'editModal']);
    }
    
    public function delete($ids)
    {
        if (!Gate::allows('delete_user') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = User::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'User has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }

    public function status($id){
        if (!Gate::allows('status_user') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $user = User::whereId($id)->first();
        if($user->status == 'Active'){
            $user->update(['status' => 'InActive']);
        }elseif($user->status == 'InActive'){
            $user->update(['status' => 'Active']);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'User Status has been updated successfully!']);
    }
    
}
