<?php

namespace App\Livewire\Admin;

use App\Models\Testimonial;
use App\Models\Language;
use App\Models\TestimonialTranslation;

use App\Helpers\Helper;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class TestimonialComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;

    public $testimonial;
    public $image;
    public $translations = [];
    public $languages;

    public $createModal = false;
    protected $listeners = ['createModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function create()
    {
        if (!Gate::allows('create_testimonial') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->testimonial = new Testimonial();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Create New Testimonial']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_testimonial') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->testimonial = Testimonial::with('translations')->findOrFail($id);
        // Initialize translations array
        $this->translations = [];

        foreach (Language::where('status', 'Active')->get() as $language) {
            $translation = $this->testimonial->translations->where('language_id', $language->id)->first();

            $this->translations[$language->id] = [
                'name'        => $translation->name ?? '',
                'review' => $translation->review ?? '',
            ];
        }
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Update Testimonial']);
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                    'image'        => $this->image,
                ], [
                    'image'        => 'nullable|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                // Dynamic translation validation rules
                foreach (Language::where('status', 'Active')->get() as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.name"   => 'required|string|max:100',
                        "translations.{$language->id}.review" => 'nullable|string|max:500',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }

                $validated = $validator->validated();

                // Handle image upload if provided
                if ($this->image && $this->image->isValid()) {
                    // If editing and  already has image — delete it
                    if ($this->testimonial->exists && $this->testimonial->image) {
                        Helper::deleteFile($this->testimonial->image);
                    }
                    // Save new image
                    $validated['image'] = Helper::saveFile('testimonials', $this->image, $w=1000, $h=700);
                    $this->reset('image');
                } else {
                    // Keep existing image if editing, or null if creating
                    $validated['image'] = $this->testimonial->image;
                }

                // Create or update the
                $this->testimonial->fill([
                    'image' => $validated['image'],
                ])->save();

                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->testimonial->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'name'   => $data['name'],
                            'review' => $data['review'] ?? null,
                        ]
                    );
                }
            });

            $this->dispatch('response', [
                'type'    => 'success',
                'message' => 'Testimonial has been ' . ($this->testimonial->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
            ]);
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_testimonial') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Testimonial::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Testimonial has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function render()
    {
        if (!Gate::allows('view_testimonial') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Testimonial::with('translations.language');
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }
        $collection = $query->paginate($this->perPage);
        $this->languages = Language::where('status', 'Active')->get();
        return view('admin.testimonials.index',compact('collection'));
    }
}