<?php

namespace App\Livewire\Admin;

use App\Models\Service;
use App\Models\Language;
use App\Models\ServiceTranslation;

use App\Helpers\Helper;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class ServiceComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;

    public $service;
    public $image;
    public $translations = [];
    public $languages;

    public $createModal = false;
    protected $listeners = ['createModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function create()
    {
        if (!Gate::allows('create_service') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->service = new Service();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Create New Service']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_service') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->service = Service::with('translations')->findOrFail($id);
        // Initialize translations array
        $this->translations = [];

        foreach (Language::where('status', 'Active')->get() as $language) {
            $translation = $this->service->translations->where('language_id', $language->id)->first();

            $this->translations[$language->id] = [
                'name'        => $translation->name ?? '',
                'review' => $translation->review ?? '',
            ];
        }
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Update Service']);
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                    'image'        => $this->image,
                ], [
                    'image'        => 'nullable|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                // Dynamic translation validation rules
                foreach (Language::where('status', 'Active')->get() as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.name"   => 'required|string|max:100',
                        "translations.{$language->id}.review" => 'nullable|string|max:500',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }

                $validated = $validator->validated();

                // Handle image upload if provided
                if ($this->image && $this->image->isValid()) {
                    // If editing and service already has image — delete it
                    if ($this->service->exists && $this->service->image) {
                        Helper::deleteFile($this->service->image);
                    }
                    // Save new image
                    $validated['image'] = Helper::saveFile('services', $this->image, $w=200, $h=200);
                    $this->reset('image');
                } else {
                    // Keep existing image if editing, or null if creating
                    $validated['image'] = $this->service->image;
                }

                // Create or update
                $this->service->fill([
                    'image' => $validated['image'],
                ])->save();

                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->service->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'name'   => $data['name'],
                            'review' => $data['review'] ?? null,
                        ]
                    );
                }
            });

            $this->dispatch('response', [
                'type'    => 'success',
                'message' => 'Service has been ' . ($this->service->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
            ]);
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_service') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Service::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Service has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function render()
    {
        if (!Gate::allows('view_service') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Service::with('translations.language');
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }
        $collection = $query->paginate($this->perPage);
        $this->languages = Language::where('status', 'Active')->get();
        return view('admin.services.index',compact('collection'));
    }
}