<?php

namespace App\Livewire\Admin;

use App\Models\Room;
use App\Models\Language;
use App\Models\RoomTranslation;
use App\Models\Facility;
use App\Models\RoomService;
use App\Models\Privilege;

use App\Helpers\Helper;
use App\Livewire\Traits\AuthorizesRequests;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class RoomComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;

    public $room,$price,$space,$facilities,$services,$privileges;
    public $photos = [];
    public $room_facilities = [];
    public $room_services = [];
    public $room_privileges = [];
    public $translations = [];

    public $createModal = false;
    protected $listeners = ['createModal','updateChoices','updateDescription'];

    public function createModal(){
        $this->createModal = false;
    }
    public function updateChoices($property, $value)
    {
        $this->$property = $value;
    }
    public function updateDescription($langId,$content)
    {
        $this->translations[$langId]['description'] = $content;
    }

    public function create()
    {
        if (!Gate::allows('create_room') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }

        $this->pull();
        $this->resetErrorBag();
        $this->room = new Room();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Create New Room']);
    }
    public function edit($id)
    {
        
        if (!Gate::allows('update_room') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->resetErrorBag();
        $this->room = Room::with('photos','translations')->findOrFail($id);
        $this->translations = [];
        $this->price = $this->room->price;
        $this->space = $this->room->space;
        $this->photos = $this->room->photos()->get()->map(function($photo) {
        return [
            'id' => $photo->id,
            'url' => url($photo->photo),
            'name' => $photo->photo
        ];
    });
        $languages = Language::select('id')->where('status', 'Active')->get();
        foreach ($languages as $language) {
            $translation = $this->room->translations->where('language_id', $language->id)->first();
            $this->translations[$language->id] = [
                'name'        => $translation->name ?? '',
                'short_description' => $translation->short_description ?? '',
                'description' => $translation->description ?? '',
                'capacity' => $translation->capacity ?? '',
            ];
        }
        $this->room_facilities = $this->room->facilities->pluck('id')->toArray();
        $this->room_services = $this->room->services->pluck('id')->toArray();
        $this->room_privileges = $this->room->privileges->pluck('id')->toArray();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Update Room','photos' => $this->photos]);
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                    'price' => $this->price,
                    'space' => $this->space,
                    'photos.*' => $this->photos
                ], [
                    'translations' => 'required|array',
                    'price'        => 'required|numeric',
                    'space'        => 'required|string|max:100',
                    'photos.*'     => 'nullable|image|mimes:jpeg,png,jpg,webp|max:5120|exclude'
                ]);
                $languages = Language::select('id')->where('status', 'Active')->get();
                // Dynamic translation validation rules
                foreach ($languages as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.name"   => 'required|string|max:100',
                        "translations.{$language->id}.short_description" => 'required|string|max:500',
                        "translations.{$language->id}.description" => 'nullable|string|max:5000',
                        "translations.{$language->id}.capacity" => 'required|string|max:200',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }
                $validated = $validator->validated();
                $slug = $this->translations[$languages[0]->id]['name'] ?? null;
                $existingCount = Room::where('slug', 'LIKE', "{$slug}%")->count();
                $validated['slug'] = $existingCount ? "{$slug}-{$existingCount}" : $slug;
                
               
                // Create or update 
                $this->room->fill([
                    'price' => $validated['price'],
                    'space' => $validated['space'],
                    'slug'  => Str::slug($validated['slug'], '-'),
                ])->save();
                // Save photos
                foreach ($this->photos as $photo) {
                    $path = Helper::saveFile('rooms',$photo,$w=1000,$h=700);
                    $this->room->photos()->create([
                        'photo' => $path
                    ]);
                }
                $this->photos = [];
                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->room->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'name'              => $data['name'] ?? null,
                            'short_description' => $data['short_description'] ?? null,
                            'description'       => $data['description'] ?? null,
                            'capacity'          => $data['capacity'] ?? null,
                        ]
                    );
                }
                $this->room->facilities()->sync($this->room_facilities);
                $this->room->services()->sync($this->room_services);
                $this->room->privileges()->sync($this->room_privileges);
                $this->dispatch('response', [
                    'type'    => 'success',
                    'message' => 'Room has been ' . ($this->room->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
                ]);
            });
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }
    public function status($id){
        if (!Gate::allows('status_room') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->room = Room::whereId($id)->first();
        if($this->room->status == 'Active'){
            $this->room->update(['status' => 'InActive']);
        }elseif($this->room->status == 'InActive'){
            $this->room->update(['status' => 'Active']);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Room Status has been updated successfully!']);
    }
    public function delete($ids)
    {
        if (!Gate::allows('delete_room') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Room::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Room has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
     public function deletePhoto($key){
        if (!Gate::allows('delete_room') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $photo = $this->room->photos()->whereId($key)->first();
        if ($photo) {
            $photo->delete();
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Photos has been deleted successfully!']);
    }

    public function render()
    {
        if (!Gate::allows('view_room') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Room::with('translations.language','photos');
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('name', 'like', '%' . $this->search . '%');
                });
            });
        }
        $collection = $query->paginate($this->perPage);
        $languages = Language::select('id','name')->where('status', 'Active')->get();
        $languagesIds = $languages->pluck('id')->toArray();
        $this->facilities = Facility::with(['translations' => function ($q) use ($languagesIds) {
            $q->whereIn('language_id', $languagesIds);
        }])->get()->map(function ($facility) {
            $names = $facility->translations->map(function($t) {
                return $t->name;
            })->join(' / ');
            return [
                'id' => $facility->id,
                'name' => $names
            ];
        })->toArray();
        $this->services = RoomService::with(['translations' => function ($q) use ($languagesIds) {
            $q->whereIn('language_id', $languagesIds);
        }])->get()->map(function ($service) {
            $names = $service->translations->map(function($t) {
                return $t->name;
            })->join(' / ');
        
            return [
                'id' => $service->id,
                'name' => $names
            ];
        })->toArray();
        $this->privileges = Privilege::with(['translations' => function ($q) use ($languagesIds) {
            $q->whereIn('language_id', $languagesIds);
        }])->get()->map(function ($privilege) {
            $names = $privilege->translations->map(function($t) {
                return $t->name;
            })->join(' / ');
        
            return [
                'id' => $privilege->id,
                'name' => $names
            ];
        })->toArray();
        return view('admin.rooms.index',compact('collection','languages'));
    }
}