<?php

namespace App\Livewire\Admin;
use App\Models\Role;
use App\Models\Module;

use Livewire\Component;
use Livewire\WithPagination;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Facades\Gate;

class RoleComponent extends Component
{
    use WithPagination, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;
    public $role, $name, $description,$modules;
    public $permissions = [];

    public $createModal = false;
    public $editModal = false;
    

    protected $listeners = ['createModal','editModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function editModal(){
        $this->editModal = false;
    }

    public function render()
    {
        if (!Gate::allows('view_role') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }

        $query = Role::with('permissions');
        if($this->search){
            $query->where('name','like', '%'.$this->search.'%');
        }
        $collection = $query->paginate($this->perPage);
        return view('admin.roles.index',compact('collection'));
    }

    public function create()
    {
        if (!Gate::allows('create_role') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->createModal = true;
        $this->modules = Module::with('permissions')->get(); 
        $this->dispatch('showModal',['modal'=>'createModal']);
    }
    public function store()
    {
        if (!Gate::allows('create_role') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'name'          =>  'required|alpha_spaces|max:50|unique:roles,name,NULL,id,deleted_at,NULL',
            'description'   =>  'nullable|string|max:500',
            'permissions'   =>  'nullable|array|exclude',
            'permissions.*' =>  'exists:permissions,id',
        ]);
        $role = Role::create($data);
        $role->permissions()->sync($this->permissions);
        $this->dispatch('response', ['type' => 'success', 'message' => 'Role has been created successfully!']);
        $this->dispatch('hideModal',['modal'=>'createModal']);
    }
    
    public function edit($id)
    {
        if (!Gate::allows('update_role') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }

        $this->editModal = true;
        $this->resetErrorBag();
        $this->role = Role::findOrFail($id);
        $this->id = $this->role->id;
        $this->name = $this->role->name;
        $this->description = $this->role->description;
        $this->modules = Module::all();
        $this->permissions = $this->role->permissions->pluck('id')->toArray();
        $this->dispatch('showModal',['modal'=>'editModal']);
    }

    public function update($id)
    {
        if (!Gate::allows('update_role') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'name'          =>  'required|alpha_spaces|max:50|unique:roles,name,'.$id.',id,deleted_at,NULL',
            'description'   =>  'nullable|string|max:500',
            'permissions'   =>  'nullable|array|exclude',
            'permissions.*' =>  'exists:permissions,id',
        ]);
        $item = Role::findOrFail($id);
        $item->update($data);
        $item->permissions()->sync($this->permissions);
        $this->dispatch('response', ['type' => 'success', 'message' => 'Role has been updated successfully!']);
        $this->dispatch('hideModal',['modal'=>'editModal']);
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_role') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Role::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Role has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
}
