<?php

namespace App\Livewire\Admin;
use App\Models\Page;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Gate;

use Livewire\Component;
use App\Livewire\Traits\AuthorizesRequests;

class PageComponent extends Component
{
    use AuthorizesRequests;
    public $search = '';
    public $perPage = 10;
    public $page, $title, $description;
   

    public $createModal = false;
    public $editModal = false;
    

    protected $listeners = ['createModal','editModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function editModal(){
        $this->editModal = false;
    }

    public function render()
    {
        if (!Gate::allows('view_page') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Page::query();
        if($this->search){
            $query->where('title','like', '%'.$this->search.'%');
        }
        $collection = $query->paginate($this->perPage);
        return view('admin.pages.index',compact('collection'));
    }
    public function create()
    {
        if (!Gate::allows('create_page') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal']);
    }
    public function store()
    {
        if (!Gate::allows('create_page') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'title'          =>  'required|alpha_spaces|max:50|unique:pages,title,NULL,id,deleted_at,NULL',
            'description'   =>  'nullable|string',
        ]);
        $data['slug'] = Str::slug($data['title'], '-');
        $page = Page::create($data);
        $this->dispatch('response', ['type' => 'success', 'message' => 'Page has been created successfully!']);
        $this->dispatch('hideModal',['modal'=>'createModal']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_page') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }

        $this->editModal = true;
        $this->resetErrorBag();
        $this->page = Page::findOrFail($id);
        $this->id = $this->page->id;
        $this->title = $this->page->title;
        $this->description = $this->page->description;
        $this->dispatch('showModal',['modal'=>'editModal']);
    }
    public function update($id)
    {
        if (!Gate::allows('update_page') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $data = $this->validate([
            'title'          =>  'required|alpha_spaces|max:50|unique:pages,title,'.$id.',id,deleted_at,NULL',
            'description'   =>  'nullable|string|max:2',
        ]);
        $data['slug'] = Str::slug($data['title'], '-');
        $item = Page::findOrFail($id);
        $item->update($data);
        $this->dispatch('response', ['type' => 'success', 'message' => 'Page has been updated successfully!']);
        $this->dispatch('hideModal',['modal'=>'editModal']);
    }
    public function delete($ids)
    {
        if (!Gate::allows('delete_page') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Page::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Page has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function status($id){
        if (!Gate::allows('status_page') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $page = Page::whereId($id)->first();
        if($page->status == 'Active'){
            $page->update(['status' => 'InActive']);
        }elseif($page->status == 'InActive'){
            $page->update(['status' => 'Active']);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Page Status has been updated successfully!']);
    }
}
