<?php

namespace App\Livewire\Admin;

use App\Models\Offer;
use App\Models\Language;
use App\Models\OfferTranslation;

use App\Helpers\Helper;
use App\Livewire\Traits\AuthorizesRequests;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

use Livewire\Component;
use Livewire\WithPagination;
use Livewire\WithFileUploads;

class OfferComponent extends Component
{
    use WithPagination, WithFileUploads, AuthorizesRequests;
    public $search = '';
    public $perPage = 10;

    public $offer,$price,$start_from,$valid_untill,$image;
    public $translations = [];
    public $languages;

    public $createModal = false;
    protected $listeners = ['createModal'];
    public function createModal(){
        $this->createModal = false;
    }
    public function create()
    {
        if (!Gate::allows('create_offer') && !Gate::allows('create_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->pull();
        $this->resetErrorBag();
        $this->offer = new Offer();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Create New Offer']);
    }
    public function edit($id)
    {
        if (!Gate::allows('update_offer') && !Gate::allows('update_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->offer = Offer::with('translations')->findOrFail($id);
        $this->price = $this->offer->price;
        $this->start_from = $this->offer->start_from;
        $this->valid_untill = $this->offer->valid_untill;
        // Initialize translations array
        $this->translations = [];

        foreach (Language::where('status', 'Active')->get() as $language) {
            $translation = $this->offer->translations->where('language_id', $language->id)->first();

            $this->translations[$language->id] = [
                'title'        => $translation->title ?? '',
                'short_description' => $translation->short_description ?? '',
                'description' => $translation->description ?? '',
            ];
        }
        $this->resetErrorBag();
        $this->createModal = true;
        $this->dispatch('showModal',['modal'=>'createModal','title' => 'Update Offer']);
    }
    public function save()
    {
        try {
            DB::transaction(function () {
                // Validate basic fields
                $validator = Validator::make([
                    'translations' => $this->translations,
                    'image'        => $this->image,
                    'price'             => $this->price,
                    'start_from'        => $this->start_from,
                    'valid_untill'      => $this->valid_untill,
                ], [
                    'price'             => 'required|numeric',
                    'start_from'        => 'nullable|date_format:Y-m-d',
                    'valid_untill'      => 'required|date_format:Y-m-d',
                    'image'             => 'nullable|mimes:jpeg,png,jpg,webp|max:5120',
                ]);
                $languages = Language::where('status', 'Active')->get();
                // Dynamic translation validation rules
                foreach ($languages as $language) {
                    $validator->addRules([
                        "translations.{$language->id}.title"   => 'required|string|max:100',
                        "translations.{$language->id}.short_description" => 'required|string|max:1000',
                        "translations.{$language->id}.description" => 'nullable|string|max:5000',
                    ]);
                }

                if ($validator->fails()) {
                    $this->setErrorBag($validator->errors());
                }

                $validated = $validator->validated();
                $slug = $this->translations[$languages[0]->id]['title'] ?? null;
                $existingCount = Offer::where('slug', 'LIKE', "{$slug}%")->count();
                $validated['slug'] = $existingCount ? "{$slug}-{$existingCount}" : $slug;
                // Handle image upload if provided
                if ($this->image && $this->image->isValid()) {
                    // If editing and already has image — delete it
                    if ($this->offer->exists && $this->offer->image) {
                        Helper::deleteFile($this->offer->image);
                    }
                    // Save new image
                    $validated['image'] = Helper::saveFile('offers', $this->image, $w=1000,$h=700);
                    $this->reset('image');
                } else {
                    // Keep existing image if editing, or null if creating
                    $validated['image'] = $this->offer->image;
                }

                // Create or update
                $this->offer->fill([
                    'price' => $validated['price'],
                    'start_from' => $validated['start_from'],
                    'valid_untill' => $validated['valid_untill'],
                    'image' => $validated['image'],
                    'slug'  => Str::slug($validated['slug'], '-'),
                ])->save();

                // Save translations
                foreach ($validated['translations'] as $language_id => $data) {
                    $this->offer->translations()->updateOrCreate(
                        ['language_id' => $language_id],
                        [
                            'title'   => $data['title'],
                            'short_description' => $data['short_description'] ?? null,
                            'description' => $data['description'] ?? null,
                        ]
                    );
                }
            });

            $this->dispatch('response', [
                'type'    => 'success',
                'message' => 'Offer has been ' . ($this->offer->wasRecentlyCreated ? 'created' : 'updated') . ' successfully!'
            ]);
            $this->dispatch('hideModal', ['modal' => 'createModal']);
        } catch (\Throwable $e) {
            report($e);
            $this->dispatch('response', [
                'type'    => 'danger',
                'message' => 'Something went wrong. Please try again!'
            ]);
        }
    }

    public function delete($ids)
    {
        if (!Gate::allows('delete_offer') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $collection = Offer::find($ids);
        if($collection->count()){
            foreach($collection as $item){
                $item->delete();
            }
        } 
        $this->dispatch('response', ['type' => 'success', 'message' => 'Offer has been deleted successfully!']);
        $this->dispatch('hideModal',['modal'=>'confirmModal']);
    }
    public function status($id){
        if (!Gate::allows('status_offer') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->offer = Offer::whereId($id)->first();
        if($this->offer->status == 'Active'){
            $this->offer->update(['status' => 'InActive']);
        }elseif($this->offer->status == 'InActive'){
            $this->offer->update(['status' => 'Active']);
        }
        $this->dispatch('response', ['type' => 'success', 'message' => 'Offer Status has been updated successfully!']);
    }
    public function render()
    {
        if (!Gate::allows('view_offer') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Offer::with('translations.language');
        if($this->search){
            $query->when($this->search, function ($q) {
                $q->whereHas('translations', function ($query) {
                    $query->where('title', 'like', '%' . $this->search . '%');
                });
            });
        $query->orWhere('price','like', '%'.$this->search.'%')
            ->orWhere('start_from','like', '%'.$this->search.'%')
            ->orWhere('valid_untill','like', '%'.$this->search.'%')
            ->orWhere('status','like', '%'.$this->search.'%');
        }
        $collection = $query->paginate($this->perPage);
        $this->languages = Language::where('status', 'Active')->get();
        return view('admin.offers.index',compact('collection'));
    }
}