<?php

namespace App\Livewire\Admin;
use App\Models\Language;
use Livewire\Component;
use Livewire\WithPagination;
use App\Livewire\Traits\AuthorizesRequests;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
class LanguageSettingsComponent extends Component
{
    use WithPagination, AuthorizesRequests;

    public $search = '';
    public $perPage = 10;

    public $language,$name,$locale,$code,$direction,$flag,$is_default,$status;
    public $flags = [];

    public $editMode = false;
    protected $listeners = ['updateChoices'];

    public function updateChoices($property, $value)
    {
        $this->$property = $value;
    }
    public function render()
    {
        if (!Gate::allows('view_languagesettings') && !Gate::allows('view_all')) {
            return $this->authorizeAndRedirect();
        }
        $query = Language::query();
        if($this->search){
            $query->where('name', 'like', '%' . $this->search. '%')
            ->orWhere('locale', 'like', '%' . $this->search. '%')
            ->orWhere('code', 'like', '%' . $this->search. '%')
            ->orWhere('direction', 'like', '%' . $this->search. '%');
        }
        $collection = $query->orderBy('id','ASC')->paginate($this->perPage);
        
        $this->flags = collect(Storage::disk('public')->files('flags'))
            ->filter(fn($file) => str_ends_with($file, '.svg'))
            ->map(function ($file) {
                $code = strtolower(pathinfo($file, PATHINFO_FILENAME)); // e.g. 'us'
                $label = locale_get_display_region('-' . strtoupper($code), 'en'); // "United States"
                return [
                    'id' => basename($file), // 'us.svg'
                    'name' => '<img src="' . asset('storage/' . $file) . '" style="width:20px; height:15px; margin-right:6px;"> ' . $label,
                ];
            })
            ->sortBy('label')->values()->toArray();
        return view('admin.language-settings-component',compact('collection'));
    }
    public function store()
    {
        $data = $this->validate([
            'name'      => 'required|string|max:50|unique:languages,name',
            'locale'    => 'required|string|max:50',
            'code'      => 'required|string|max:10|unique:languages,code',
            'direction' => 'required|in:rtl,ltr',
            'flag'      => 'required|string|max:50',
        ]);
        
        Language::create($data);
        $this->pull();
        $this->resetErrorBag();
        return $this->dispatch('response', ['type' => 'success', 'message' => 'Language has been created successfully!']);
    }
    public function edit($id)
    {
        $this->language = Language::findOrFail($id);
        $this->name = $this->language->name;
        $this->code = $this->language->code;
        $this->locale = $this->language->locale;
        $this->is_default = $this->language->is_default;
        $this->direction = $this->language->direction;
        $this->status = $this->language->status;
        $this->flag = basename($this->language->flag);
        $this->editMode = true;
    }
    public function update()
    {
        $data = $this->validate([
            'name'      =>  'required|string|max:50|unique:languages,name,'.$this->language->id.',id,deleted_at,NULL',
            'locale'    => 'required|string|max:50',
            'code'      => 'required|string|max:10|unique:languages,code,' . $this->language->id. ',id,deleted_at,NULL',
            'direction' => 'required|in:rtl,ltr',
            'flag'      => 'required|string|max:50',
        ]);
        $data['flag'] = 'storage/flags/'.$this->flag;
        $this->language->update($data);
        return $this->dispatch('response', ['type' => 'success', 'message' => 'Language has been updated successfully!']);
    }
    public function delete($id)
    {
        if (!Gate::allows('delete_languagesettings') && !Gate::allows('delete_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->language = Language::findOrFail($id);

        // Optional: Don't delete default language
        if ($this->language->is_default === 'Yes') {
            return $this->dispatch('response', ['type' => 'success', 'message' => 'Cannot delete the default language!']);
        }

        $this->language->delete();
        session()->flash('success', 'Language deleted.');
    }
    public function updateStatus($id){
        if (!Gate::allows('status_languagesettings') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->language = Language::findOrFail($id);
        $this->language->status = $this->language->status === 'Active' ? 'Inactive' : 'Active';
        $this->language->save();
        return $this->dispatch('response', ['type' => 'success', 'message' => 'Language Status has been updated successfully!']);
    }
    public function isDefault($id){
        if (!Gate::allows('status_languagesettings') && !Gate::allows('status_all')) {
            return $this->authorizeAndRedirect();
        }
        $this->language = Language::findOrFail($id);
        // Ensure the language is Active before making it default
        if ($this->language->status === 'Inactive') {
            $this->language->status = 'Active';
        }
        $language = $this->language;
        DB::transaction(function () use ($language) {
            // Set all others to is_default = 'No'
            Language::query()->update(['is_default' => 'No']);
            // Set this one to is_default = 'Yes' and save any status change
            $language->is_default = 'Yes';
            $language->save();
        });
        return $this->dispatch('response', ['type' => 'success', 'message' => 'Language default Status has been updated successfully!']);
    }
}
